<?php
declare(strict_types=1);

use App\Domain\Link\LinkRepository;
use App\Domain\Rule\RuleEngine;
use App\Http\Controller\RedirectController;
use App\Http\Middleware\RateLimiterMiddleware;
use App\Security\UrlSafetyValidator;
use App\Support\Database;
use App\Support\RedisFactory;
use GuzzleHttp\Client;
use Monolog\Handler\StreamHandler;
use Monolog\Logger;
use Psr\Log\LoggerInterface;
use Symfony\Component\Messenger\MessageBus;
use Symfony\Component\Messenger\Handler\HandlersLocator;
use Symfony\Component\Messenger\Middleware\HandleMessageMiddleware;
use Symfony\Component\Serializer\Serializer;
use Symfony\Component\Serializer\Normalizer\ObjectNormalizer;

return [
    PDO::class => function (): PDO {
        $dsn = $_ENV['DB_DSN'] ?? '';
        $user = $_ENV['DB_USER'] ?? '';
        $pass = $_ENV['DB_PASS'] ?? '';
        return Database::make($dsn, $user, $pass);
    },

    Predis\Client::class => function (): Predis\Client {
        $url = $_ENV['REDIS_URL'] ?? 'redis://127.0.0.1:6379';
        return RedisFactory::make($url);
    },

    LoggerInterface::class => function (): LoggerInterface {
        $logger = new Logger('app');
        $logger->pushHandler(new StreamHandler(__DIR__ . '/../var/log/app.log'));
        return $logger;
    },

    Client::class => function (): Client {
        return new Client([
            'timeout' => 1.5,
            'http_errors' => false,
        ]);
    },

    UrlSafetyValidator::class => function (Client $http, Predis\Client $redis): UrlSafetyValidator {
        return new UrlSafetyValidator($http, $redis, $_ENV['SAFE_BROWSING_KEY'] ?? '');
    },

    RuleEngine::class => DI\factory(function (Predis\Client $redis, Client $http, UrlSafetyValidator $safety) {
        $vpnChecker = function (string $ip) use ($redis, $http): bool {
            $cacheKey = 'vpn:' . $ip;
            $cached = $redis->get($cacheKey);
            if ($cached !== null) {
                return $cached === '1';
            }
            // Extremely conservative: treat upstream failures as NOT vpn to avoid false positives.
            $isVpn = false;
            try {
                $resp = $http->get('https://blackbox.ipinfo.app/lookup/' . rawurlencode($ip));
                if ($resp->getStatusCode() === 200) {
                    $txt = trim((string) $resp->getBody());
                    $isVpn = ($txt === 'Y');
                }
            } catch (Throwable $e) {
                $isVpn = false;
            }
            $redis->setex($cacheKey, 86400, $isVpn ? '1' : '0');
            return $isVpn;
        };

        return new RuleEngine($vpnChecker, $safety);
    }),

    LinkRepository::class => DI\autowire(),

    RedirectController::class => DI\autowire(),

    RateLimiterMiddleware::class => DI\autowire(),

    MessageBus::class => function () {
        $locator = new HandlersLocator([]); // Handlers wired later if needed
        $middleware = [new HandleMessageMiddleware($locator)];
        return new MessageBus($middleware);
    },

    Serializer::class => function () {
        return new Serializer([new ObjectNormalizer()]);
    },
];
