<?php
declare(strict_types=1);

namespace App\Domain\Link;

use PDO;

final class LinkRepository
{
    public function __construct(private PDO $pdo)
    {
    }

    public function findByCode(string $code): ?array
    {
        $sql = "SELECT id, code, target_url, fallback_url, country_allow, device, flags
                FROM links WHERE code = :code AND is_active = 1 LIMIT 1";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([':code' => $code]);
        $row = $stmt->fetch();
        if (!$row) {
            return null;
        }
        // JSON fields to arrays
        foreach (['country_allow', 'flags'] as $col) {
            if (isset($row[$col]) && is_string($row[$col])) {
                $row[$col] = json_decode($row[$col], true) ?: [];
            }
        }
        return $row;
    }

    public function logClick(int $linkId, string $ip, string $ua, string $country, string $device, bool $isVpn): void
    {
        $sql = "INSERT INTO clicks(link_id, ip, user_agent, country, device, is_vpn) 
                VALUES (:link_id, INET6_ATON(:ip), :ua, :country, :device, :is_vpn)";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([
            ':link_id' => $linkId,
            ':ip' => $ip,
            ':ua' => $ua,
            ':country' => strtolower($country),
            ':device' => $device,
            ':is_vpn' => $isVpn ? 1 : 0,
        ]);
    }
}
