<?php
declare(strict_types=1);

namespace App\Domain\Rule;

use App\Security\UrlSafetyValidator;
use Detection\MobileDetect;

final class RuleDecision
{
    public function __construct(
        public bool $useFallback,
        public string $url
    ) {}
}

final class RuleEngine
{
    /** @var callable(string):bool */
    private $vpnChecker;
    private UrlSafetyValidator $safety;
    private MobileDetect $detect;

    /**
     * @param callable(string):bool $vpnChecker
     */
    public function __construct(callable $vpnChecker, UrlSafetyValidator $safety = null)
    {
        $this->vpnChecker = $vpnChecker;
        $this->safety = $safety ?? new UrlSafetyValidator();
        $this->detect = new MobileDetect();
    }

    /**
     * @param array{country_allow?:array, device?:string, target_url:string, fallback_url:string} $rules
     */
    public function decide(array $rules, string $ip, string $ua, string $country, bool $isWap, string $traffic): RuleDecision
    {
        $target = $rules['target_url'];
        $fallback = $rules['fallback_url'];

        // Country allow-list
        $allow = array_map('strtolower', $rules['country_allow'] ?? []);
        if ($allow && !in_array(strtolower($country), $allow, true)) {
            return new RuleDecision(true, $fallback);
        }

        // Device filter
        $device = strtolower($rules['device'] ?? 'any');
        $this->detect->setUserAgent($ua);
        $isMobile = $this->detect->isMobile() || $this->detect->isTablet();
        if ($device === 'wap' && !$isMobile) {
            return new RuleDecision(true, $fallback);
        }
        if ($device === 'web' && $isMobile) {
            return new RuleDecision(true, $fallback);
        }

        // VPN/Proxy check (cached)
        $isVpn = call_user_func($this->vpnChecker, $ip);
        if ($isVpn) {
            return new RuleDecision(true, $fallback);
        }

        // URL safety
        if (!$this->safety->isSafe($target)) {
            return new RuleDecision(true, $fallback);
        }

        return new RuleDecision(false, $target);
    }
}
