<?php
declare(strict_types=1);

namespace App\Http\Controller;

use App\Domain\Link\LinkRepository;
use App\Domain\Rule\RuleEngine;
use Nyholm\Psr7\Response;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Http\Server\RequestHandlerInterface;

final class RedirectController implements RequestHandlerInterface
{
    public function __construct(
        private LinkRepository $links,
        private RuleEngine $rules
    ) {}

    public function handle(ServerRequestInterface $request): ResponseInterface
    {
        $args = $request->getAttribute('route')->getArguments();
        $code = (string) ($args['code'] ?? '');
        if ($code === '') {
            return new Response(404);
        }

        $ip = $request->getServerParams()['REMOTE_ADDR'] ?? '0.0.0.0';
        $ua = $request->getHeaderLine('User-Agent');

        $country = strtolower($request->getHeaderLine('Cf-Ipcountry') ?: 'xx');
        $traffic = $request->getHeaderLine('Sec-CH-UA-Platform') ? 'web' : 'web';
        $isWap = false; // derived inside RuleEngine via UA

        $meta = $this->links->findByCode($code);
        if ($meta === null) {
            return new Response(404);
        }

        $decision = $this->rules->decide($meta, $ip, $ua, $country, $isWap, $traffic);

        $location = $decision->url;
        $status = 302;

        $resp = new Response($status);
        $resp = $resp
            ->withHeader('Location', $location)
            ->withHeader('Cache-Control', 'private, max-age=60')
            ->withHeader('Content-Security-Policy', "default-src 'none'; frame-ancestors 'none'; base-uri 'none';")
            ->withHeader('X-Frame-Options', 'DENY')
            ->withHeader('Referrer-Policy', 'no-referrer')
            ->withHeader('Strict-Transport-Security', 'max-age=31536000; includeSubDomains; preload');

        return $resp;
    }
}
