<?php
declare(strict_types=1);

namespace App\Security;

use GuzzleHttp\Client;
use Predis\Client as Redis;

final class UrlSafetyValidator
{
    public function __construct(
        private ?Client $http = null,
        private ?Redis $redis = null,
        private string $googleKey = ''
    ) {}

    public function isSafe(string $url): bool
    {
        // Very conservative: if key missing, treat as safe to avoid blocking legitimate flows.
        if (!$this->googleKey || !$this->http) {
            return true;
        }
        $cacheKey = 'safe:' . sha1($url);
        if ($this->redis && ($hit = $this->redis->get($cacheKey)) !== null) {
            return $hit === '1';
        }
        try {
            $resp = $this->http->post('https://safebrowsing.googleapis.com/v4/threatMatches:find?key=' . $this->googleKey, [
                'json' => [
                    'client' => ['clientId' => 'smart-redirect', 'clientVersion' => '1.0'],
                    'threatInfo' => [
                        'threatTypes' => ['MALWARE','SOCIAL_ENGINEERING','UNWANTED_SOFTWARE','POTENTIALLY_HARMFUL_APPLICATION'],
                        'platformTypes' => ['ANY_PLATFORM'],
                        'threatEntryTypes' => ['URL'],
                        'threatEntries' => [['url' => $url]],
                    ],
                ],
            ]);
            $data = json_decode((string) $resp->getBody(), true) ?: [];
            $safe = empty($data['matches']);
        } catch (\Throwable $e) {
            $safe = true; // fail-open to reduce false positives, edge WAF should catch worst cases
        }
        if ($this->redis) {
            $this->redis->setex($cacheKey, 86400, $safe ? '1' : '0');
        }
        return $safe;
    }
}
